// js/crypto-js.js
// Classic PKZIP "ZipCrypto" implementation (no AES).
// Compatible with Windows-style ZipCrypto (intended for Explorer & 7-Zip).
//
// Exposes:
//   window.ZipCrypto = {
//     encrypt(dataUint8, password, crc32Value) -> Uint8Array   // header+encrypted
//     decrypt(encryptedUint8, password)       -> Uint8Array   // decrypted data
//   }

(function (global) {
  'use strict';

  const CRC_TABLE = (() => {
    const t = new Uint32Array(256);
    for (let i = 0; i < 256; i++) {
      let c = i;
      for (let k = 0; k < 8; k++) {
        c = (c & 1) ? (0xEDB88320 ^ (c >>> 1)) : (c >>> 1);
      }
      t[i] = c >>> 0;
    }
    return t;
  })();

  function crc32UpdateByte(crc, b) {
    return (CRC_TABLE[(crc ^ b) & 0xFF] ^ (crc >>> 8)) >>> 0;
  }

  function initKeys(password) {
    let key0 = 0x12345678 >>> 0;
    let key1 = 0x23456789 >>> 0;
    let key2 = 0x34567890 >>> 0;

    const pwBytes = new TextEncoder().encode(String(password));
    for (let i = 0; i < pwBytes.length; i++) {
      const c = pwBytes[i];
      key0 = crc32UpdateByte(key0, c);
      key1 = (key1 + (key0 & 0xFF)) >>> 0;
      key1 = (key1 * 134775813 + 1) >>> 0;
      key2 = crc32UpdateByte(key2, (key1 >>> 24) & 0xFF);
    }
    return { key0, key1, key2 };
  }

  function updateKeys(keys, plainByte) {
    keys.key0 = crc32UpdateByte(keys.key0, plainByte);
    keys.key1 = (keys.key1 + (keys.key0 & 0xFF)) >>> 0;
    keys.key1 = (keys.key1 * 134775813 + 1) >>> 0;
    keys.key2 = crc32UpdateByte(keys.key2, (keys.key1 >>> 24) & 0xFF);
  }

  function decryptByte(keys, c) {
    const temp = (keys.key2 | 2) & 0xFFFF;
    const xor = ((temp * (temp ^ 1)) >>> 8) & 0xFF;
    const plain = (c ^ xor) & 0xFF;
    updateKeys(keys, plain);
    return plain;
  }

  function encryptByte(keys, plain) {
    const temp = (keys.key2 | 2) & 0xFFFF;
    const xor = ((temp * (temp ^ 1)) >>> 8) & 0xFF;
    const out = (plain ^ xor) & 0xFF;
    updateKeys(keys, plain);
    return out;
  }

  function encrypt(data, password, crc32Value) {
    if (!password) return data;

    const keys = initKeys(password);
    const header = new Uint8Array(12);
    const random11 = crypto.getRandomValues(new Uint8Array(11));

    for (let i = 0; i < 11; i++) {
      header[i] = encryptByte(keys, random11[i]);
    }

    // Traditional check byte: high-order byte of CRC32
    const checkByte = (crc32Value >>> 24) & 0xFF;
    header[11] = encryptByte(keys, checkByte);

    const out = new Uint8Array(12 + data.length);
    out.set(header, 0);

    let o = 12;
    for (let i = 0; i < data.length; i++) {
      out[o++] = encryptByte(keys, data[i]);
    }
    return out;
  }

  function decrypt(encrypted, password) {
    if (!password) return encrypted;
    if (!encrypted || encrypted.length < 12) {
      throw new Error('Encrypted data too short');
    }

    const keys = initKeys(password);
    let i = 0;

    for (; i < 12; i++) {
      decryptByte(keys, encrypted[i]);
    }

    const out = new Uint8Array(encrypted.length - 12);
    let o = 0;
    for (; i < encrypted.length; i++) {
      out[o++] = decryptByte(keys, encrypted[i]);
    }
    return out;
  }

  global.ZipCrypto = { encrypt, decrypt };
})(typeof self !== 'undefined' ? self : window);
